# Build macOS app on GitHub Actions
# This triggers a workflow run and downloads the result

$Owner = "MiBaDK"
$Repo = "video-manager"

$ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$ProjectDir = Split-Path -Parent $ScriptDir
$OutputDir = Join-Path $ProjectDir "dist"

if (-not (Test-Path $OutputDir)) {
    New-Item -ItemType Directory -Path $OutputDir | Out-Null
}

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "MiBa Video Manager - macOS Build" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

# Trigger workflow
Write-Host "Triggering macOS build on GitHub Actions..." -ForegroundColor Yellow

$WorkflowId = "build-macos.yml"
$Response = gh api "repos/$Owner/$Repo/actions/workflows/$WorkflowId/dispatches" `
    --method POST `
    --field ref="main"

if ($LASTEXITCODE -ne 0) {
    Write-Host "Failed to trigger workflow" -ForegroundColor Red
    exit 1
}

Write-Host "Workflow triggered! Waiting for build to start..." -ForegroundColor Green
Start-Sleep -Seconds 3

# Get latest run
$Runs = gh api "repos/$Owner/$Repo/actions/workflows/$WorkflowId/runs" | ConvertFrom-Json
$Run = $Runs.workflow_runs | Select-Object -First 1

if (-not $Run) {
    Write-Host "Could not find workflow run" -ForegroundColor Red
    exit 1
}

$RunId = $Run.id
Write-Host "Run started: $RunId" -ForegroundColor Cyan

# Wait for completion
Write-Host "Waiting for build to complete..." -ForegroundColor Yellow
$Status = $Run.status

while ($Status -eq "in_progress" -or $Status -eq "queued") {
    Start-Sleep -Seconds 10
    $Runs = gh api "repos/$Owner/$Repo/actions/workflows/$WorkflowId/runs" | ConvertFrom-Json
    $Run = $Runs.workflow_runs | Select-Object -First 1
    $Status = $Run.status
    $Conclusion = $Run.conclusion
    
    if ($Status -eq "completed") {
        break
    }
    
    Write-Host "  Status: $Status" -ForegroundColor Gray
}

Write-Host ""
Write-Host "Build status: $Status - $Conclusion" -ForegroundColor $(if ($Conclusion -eq "success") { "Green" } else { "Red" })

if ($Conclusion -ne "success") {
    Write-Host "Build failed. Check GitHub Actions for details." -ForegroundColor Red
    exit 1
}

# Download artifact
Write-Host "Downloading build..." -ForegroundColor Yellow

$Token = (gh auth token)
$Headers = @{
    "Authorization" = "Bearer $Token"
    "Accept" = "application/vnd.github+json"
}

$ArtifactsUrl = "https://api.github.com/repos/$Owner/$Repo/actions/runs/$RunId/artifacts"
$Artifacts = Invoke-RestMethod -Uri $ArtifactsUrl -Headers $Headers
$Artifact = $Artifacts.artifacts | Where-Object { $_.name -eq "MiBa-Video-Manager" }

if ($Artifact) {
    $OutputFile = Join-Path $OutputDir "MiBa-Video-Manager.zip"
    $DownloadHeaders = @{
        "Authorization" = "Bearer $Token"
        "Accept" = "application/vnd.github+json"
    }
    Invoke-WebRequest -Uri $Artifact.archive_download_url -Headers $DownloadHeaders -OutFile $OutputFile -UseBasicParsing
    Write-Host "Done! Downloaded to: $OutputFile" -ForegroundColor Green
    Write-Host ""
    Write-Host "Extract the ZIP and run the app!" -ForegroundColor Cyan
} else {
    Write-Host "Artifact not found" -ForegroundColor Red
    exit 1
}
